#ifndef DIAGS__I2C_H__LOADED
#define DIAGS__I2C_H__LOADED
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* I2C System management bus access routines */
/* Due to the flexibility of the ways in which a system may implement an
 * i2c subsystem, this code is a platform-specific thing, and interlinks
 * with lower-level access mechanisms in Northbridge or Southbridge */


/*--------------------------------------------------------------------*/
/* Data structures and definitions */

#include "i2c/i2c_device_classes.h"	/* pick up the supported device codes */

typedef signed long i2c_data;
#define I2C_INVALID_REQUEST -1		/* -1 indicates read failure */


/* I2C recognised device IDs - likely targets that could exist on an I2C bus.
 * The concept is that each system contains AT MOST one of each device */

typedef enum _i2c_device {

        /* misc Slot-B devices for up to four CPUs */
        I2C_SLOTB0_REV,          /* Slot-B Revision EEPROM */
        I2C_SLOTB1_REV,          /* Slot-B Revision EEPROM */
        I2C_SLOTB2_REV,          /* Slot-B Revision EEPROM */
        I2C_SLOTB3_REV,          /* Slot-B Revision EEPROM */

        I2C_SLOTB0_THERM,        /* Slot-B thermal sensor */
        I2C_SLOTB1_THERM,        /* Slot-B thermal sensor */
        I2C_SLOTB2_THERM,        /* Slot-B thermal sensor */
        I2C_SLOTB3_THERM,        /* Slot-B thermal sensor */

        /* motherboard devices */
        I2C_MOBO_REV,           /* Motherboard Revision EEPROM */
        I2C_SMD0,               /* system management device */
        I2C_SMD1,               /* system management device */
        I2C_SMD2,               /* system management device */
        I2C_SMD3,               /* system management device */
        I2C_SMD4,               /* system management device */
        I2C_SMD5,               /* system management device */
        I2C_CLK_BUFFER,		/* ICS9179-05 SDRAM clock buffer */
        I2C_LED,                /* LED controller */
	I2C_SYSFAN1,		/* Case fan 1 */
	I2C_SYSFAN2,		/* Case fan 2 */

	I2C_REG0,		/* Generic control registers */
	I2C_REG1,
	I2C_REG2,
	I2C_REG3,

        /* bus multiplexers (note, these should be internally managed) */
        I2C_MUX0,               /* Sub-bus access multiplexer */
        I2C_MUX1,               /* Sub-bus access multiplexer */
        I2C_MUX2,               /* Sub-bus access multiplexer */
        I2C_MUX3,               /* Sub-bus access multiplexer */

        /* keep these consecutive so they may be looped over iteratively */
        I2C_DIMM0,              /* DIMM slot 0 */
        I2C_DIMM1,              /* DIMM slot 1 */
        I2C_DIMM2,              /* DIMM slot 2 */
        I2C_DIMM3,              /* DIMM slot 3 */
        I2C_DIMM4,              /* DIMM slot 4 */
        I2C_DIMM5,              /* DIMM slot 5 */
        I2C_DIMM6,              /* DIMM slot 6 */
        I2C_DIMM7,              /* DIMM slot 7 */
        I2C_DIMM8,              /* DIMM slot 8 */
        I2C_DIMM9,              /* DIMM slot 9 */
        I2C_DIMM10,             /* DIMM slot 10 */
        I2C_DIMM11,             /* DIMM slot 11 */
        I2C_DIMM12,             /* DIMM slot 12 */
        I2C_DIMM13,             /* DIMM slot 13 */
        I2C_DIMM14,             /* DIMM slot 14 */
        I2C_DIMM15,             /* DIMM slot 15 */

        I2C_NODEV               /* used as an end-of-list marker */
} i2c_device;


/* Recognised I2C sensors for which we have drivers */

typedef enum _i2c_chip {
	I2CCHIP_PCF8574,	/* Philips Semi PCF8574 Mux/Reg */
	I2CCHIP_PCF8582,	/* Philips Semi PCF8582 256-byte EEPROM */
	I2CCHIP_ADM9240,	/* Analog Devices ADM9240 System management */
	I2CCHIP_LM75		/* Nat Semi LM75 thermal monitor */
} i2c_chip;


/* Recognised quantity measured by this component */
/* Sensors with multiple measurement channels, should be listed once for every
 * measurement quantity they are monitoring.
 * If a sensor measures multiple instances of the same quantity, they are 
 * enumerated.
 */

typedef enum _i2c_sensor {
	I2CSENS_REGMUX,
	I2CSENS_EPROM,
	I2CSENS_TEMP0,
	I2CSENS_TEMP1,
	I2CSENS_FAN0,
	I2CSENS_FAN1,
	I2CSENS_2_5V,
	I2CSENS_3_3V,
	I2CSENS_5V,
	I2CSENS_12V,
	I2CSENS_VCCP1,
	I2CSENS_VCCP2,
 	I2CSENS_CI,		/* Chassis intrusion */
} i2c_sensor;

/* Sensor basic data structures, a set of data types shared between different
 * chips measuring the same basic quantity
 */

#define I2C_TEMP_SCALE	10
typedef struct {
    uint16 temp;		/* measured temperature x10 */
    uint16 max;			/* max temperature x10 */
    uint16 hyst;		/* hysteresis temperature x10 */
} i2c_therm_t;

typedef struct {
   int rpm;			/* fan speed in rpm */
   int threshold;		/* threshold RPM below which fan is bad */
} i2c_fan_t;


/* nb voltages can be signed.
 * Bodge warning: A value of 0 overrides to be the max value on the sensor scale
 */
typedef struct {
    int v;			/* measured voltage x10 */
    int max, min;		/* max, min bounds x10 */
    int nominal;		/* override for nominal value (0 otherwise) */
} i2c_volt_t;



/* I2C platform-specific bus layout description.  Each platform should 
 * provide a null-terminated array of these.
 */

typedef struct {
    i2c_device dev;
    i2c_sensor sensor;
    i2c_chip chip;
    uint8 addr, config;
    String desc;
} I2Cbus_t;


/*----------------------------------------------------------------------*/
/* High level access routines */

/* Print device-dependent status (eg temp, fan RPM) to the buffer supplied */
DBM_STATUS i2c_status( const I2Cbus_t *I, String buf );


/* Check all I2C sensors are operating within their specified parameters */
DBM_STATUS i2c_post( void );
/* If there is an error or warning, POST should setup a console and then call
 * this routine to report the details */
void i2c_post_report( void );


const I2Cbus_t *i2c_lookup( const i2c_device D );


/*----------------------------------------------------------------------*/
/* I2C Bus controller access routines */

DBM_STATUS i2cdev_init( void );
DBM_STATUS i2cdev_read( const uint8 node, const int cmd,
			const size_t bytes, uint8 *data);
DBM_STATUS i2cdev_write( const uint8 node, const int cmd, 
			const size_t bytes, const uint8 *data);

/*----------------------------------------------------------------------*/
#endif		/*  DIAGS__I2C_H__LOADED  */
